<?php
namespace app;

use think\db\exception\DataNotFoundException;
use think\db\exception\ModelNotFoundException;
use think\exception\Handle;
use think\exception\HttpException;
use think\exception\HttpResponseException;
use think\exception\ValidateException;
use think\Response;
use Throwable;

/**
 * 应用异常处理类
 */
class ExceptionHandle extends Handle
{
    /**
     * 不需要记录信息（日志）的异常类列表
     * @var array
     */
    protected $ignoreReport = [
        HttpException::class,
        HttpResponseException::class,
        ModelNotFoundException::class,
        DataNotFoundException::class,
        ValidateException::class,
    ];

    /**
     * 记录异常信息（包括日志或者其它方式记录）
     *
     * @access public
     * @param  Throwable $exception
     * @return void
     */
    public function report(Throwable $exception): void
    {
        // 使用内置的方式记录异常日志
        parent::report($exception);
    }

    /**
     * Render an exception into an HTTP response.
     *
     * @access public
     * @param \think\Request   $request
     * @param Throwable $e
     * @return Response
     */
    public function render($request, Throwable $e): Response
    {
        // AJAX请求返回JSON格式错误信息
        if ($request->isAjax()) {
            $data = [
                'code' => $e->getCode() ?: 500,
                'msg'  => $e->getMessage(),
                'data' => [],
                'file' => $e->getFile(),
                'line' => $e->getLine()
            ];
            
            // 开发环境下输出调试信息
            if (env('APP_DEBUG')) {
                $data['trace'] = $e->getTrace();
            }
            
            return json($data)->header([
                'Access-Control-Allow-Origin' => '*',
                'Access-Control-Allow-Methods' => 'GET,POST,PATCH,PUT,DELETE,OPTIONS',
                'Access-Control-Allow-Headers' => 'Authorization,Content-Type,X-Requested-With,Accept'
            ]);
        }
        
        // 非AJAX请求显示错误页面
        if (env('APP_DEBUG')) {
            return parent::render($request, $e);
        }
        
        // 生产环境显示友好的错误页面
        return Response::create(view('error/500', [
            'message' => $e->getMessage()
        ]));
    }
}
